//************************************************************************
//                        DS1821_1.C
//                          V1.10
//  Programmer for configuring the Dallas DS1821 Thermostat devices
//  Hardware consists of a LCD driven by a PCF8577P I2C Driver, two push
//  Button switches (Select and Change), a couple of transistors for
//  selecting the power supply mode for the DS1821, a socket to accept
//  a ds1821 directly on the board, an external connector for flying leads,
//  a battery and voltage regulator, a few misc resistors, caps, etc and
//  a PIC12C509 to control the whole thing.
//
// Author: Michael Pearce
//       Chemistry Department, University of Canterbury.
//
// Started: 28 May 1999
//
//************************************************************************
//                       VERSION INFORMATION
//************************************************************************
// Version 1.20 -  June 1999
//  Altered the methods for chaning the data to a Simpler (slower) but
//  Hopefully operational system.
//  +/- keys change the total value
//  Menu Key selects High/Low/Program options
//
//************************************************************************
// Version 1.10 - 3 June 1999
//  Scraped the original menu/data selection method and started again.
//
//************************************************************************
// Version 1.00 - 28 May 1999
//   Start of the project.
//************************************************************************

#include <pic.h>

#define MAXTEMP  120
#define MINTEMP  (0-55)

#define LCD_KEYPAD            //-- If using on the LCDKEYPAD Board

#define	XTAL_FREQ	8MHZ

#define BITNUM(adr, bit)       ((unsigned)(&adr)*8+(bit))

#ifdef LCD_KEYPAD
 #define PORT PORTD
 #define TRIS TRISD
#else
 #ifdef _16C64
  #define PORT PORTB
  #define TRIS TRISB
 #endif
#endif


//************ DEFINE OTHER I/O PORTS ****************
// SCL    = PORTB/GPIO 0
// SDA    = PORTB/GPIO 1
// SW1    = PORTB/GPIO 2
// SW2    = PORTB/GPIO 3
// selpwr = PORTB/GPIO 4
// 1wire  = PORTB/GPIO 5

#ifdef _12C50x
 static bit SCL      @ BITNUM(GPIO,0);
 static bit SCL_TRIS @ BITNUM(TRIS,0);
 static bit SDA      @ BITNUM(GPIO,1);
 static bit SDA_TRIS @ BITNUM(TRIS,1);
 static bit SW1      @ BITNUM(GPIO,2);
 static bit SW1_DIR  @ BITNUM(TRIS,2);
 static bit SW2      @ BITNUM(GPIO,3);
 static bit SW2_DIR  @ BITNUM(TRIS,3);
 static bit SPWR     @ BITNUM(GPIO,4);
 static bit SPWR_DIR @ BITNUM(TRIS,4);
 static bit D_PIN    @ BITNUM(GPIO,5);
 static bit D_TRIS   @ BITNUM(TRIS,5);
#endif
#ifdef _16C64
 static bit SCL      @ BITNUM(PORT,0);
 static bit SCL_TRIS @ BITNUM(TRIS,0);
 static bit SDA      @ BITNUM(PORT,1);
 static bit SDA_TRIS @ BITNUM(TRIS,1);
 static bit SW1      @ BITNUM(PORT,2);
 static bit SW1_DIR  @ BITNUM(TRIS,2);
 static bit SW2      @ BITNUM(PORT,3);
 static bit SW2_DIR  @ BITNUM(TRIS,3);
 static bit SPWR     @ BITNUM(PORT,4);
 static bit SPWR_DIR @ BITNUM(TRIS,4);
 static bit D_PIN    @ BITNUM(PORT,5);
 static bit D_TRIS   @ BITNUM(TRIS,5);
#endif

#ifdef _16F84
 static bit SCL      @ BITNUM(PORTA,0);
 static bit SCL_TRIS @ BITNUM(TRISA,0);
 static bit SDA      @ BITNUM(PORTA,1);
 static bit SDA_TRIS @ BITNUM(TRISA,1);
 static bit SW1      @ BITNUM(PORTA,2);
 static bit SW1_DIR  @ BITNUM(TRISA,2);
 static bit SW2      @ BITNUM(PORTA,3);
 static bit SW2_DIR  @ BITNUM(TRISA,3);
 static bit SPWR     @ BITNUM(PORTB,4);
 static bit SPWR_DIR @ BITNUM(TRISB,4);
 static bit D_PIN    @ BITNUM(PORTB,5);
 static bit D_TRIS   @ BITNUM(TRISB,5);
#endif


#include "delay.h"
#include "delay.c"

#include "m_i2c_1.c"
#include "1wire002.c"


//************ DEFINE CHARACTERS *****************
                         //          A
#define LA   0x80        //        ****
#define LB   0x40        //     F *    *  B
#define LC   0x20        //       *  G *
#define LD   0x10        //        ****
#define LE   0x08        //     E *    *  C
#define LF   0x04        //       *    *
#define LG   0x02        //        ****   * DOT
#define LDOT 0x01        //          D
#define LBLANK 0x00


#define NUM0  LA+LB+LC+LD+LE+LF    //-- '0'
#define NUM1  LB+LC                //-- '1'
#define NUM2  LA+LB+LG+LE+LD       //-- '2'
#define NUM3  LA+LB+LC+LD+LG       //-- '3'
#define NUM4  LF+LG+LB+LC          //-- '4'
#define NUM5  LA+LF+LG+LC+LD       //-- '5'
#define NUM6  LA+LF+LE+LD+LC+LG    //-- '6'
#define NUM7  LA+LB+LC             //-- '7'
#define NUM8  LA+LB+LC+LD+LE+LF+LG //-- '8'
#define NUM9  LA+LB+LC+LF+LG       //-- '9'

#define MINUS LG                   //-- '-'      - Negative
#define THIGH LB+LC+LE+LF+LG       //-- 'H'      - High Level
#define TLOW  LF+LE+LD             //-- 'L'      - Low Level
#define PROGP LF+LE+LG+LB+LA       //-- 'P'      - Program
#define PROGR LG+LE                //-- 'r'      - r
#define PROGO LG+LC+LD+LE          //-- 'o'      - o
#define PROGG LA+LB+LC+LD+LF+LG    //-- 'g'      - g
#define CLRC  LG+LE+LD             //-- 'c'      -Clear
#define CLRL  LF+LE                //-- 'l'
#define CLRR  LE+LG                //-- 'r'
#define ERROR LA+LF+LG+LE+LD       //-- 'E'      - Error
#define PRGDONE LA+LG+LD           //-- 3 bars one above the other
#define DOT   LDOT     //-- '.' The Dot Shows current Position for altering

const char Digits[]={NUM0,NUM1,NUM2,NUM3,NUM4,NUM5,NUM6,NUM7,NUM8,NUM9};


//************ DEFINE TIMINGS ********************
#define _TPC    2             //-- Ms for Power Low before pulses (MIN)
#define _TCP    1             //-- Ms for toggle clock 16  (MIN)
#define _TCL    1             //-- Ms for toggle clock pulse low (MIN)
#define _TCH    1             //-- Ms for toggle clock pulse High (MIN)
#define _TT     1             //-- Ms for Transition time (ESTIMATED)


//*********** DEFINE KEY CODES *******************
#define KEY_NONE    0     //-- No keys pressed
#define KEY_MINUS   1     //-- Minus Key Only
#define KEY_PLUS    2     //-- Plus key only
#define KEY_SELECT  3     //-- Select key or both Plus and Minus keys pressed

//************ MENU TYPES **************
#define T_LOW       0
#define T_HIGH      1
#define T_CLRL      2
#define T_CLRH      3
#define T_PROG      4

#define T_NEGATIVE  2     //-- Indicates a Negative Value

//*********** Status Flag Settings ************
#define STATUS_T_H_C (4+2+0)  //-- Thermostat mode - Active High - Continuous
#define STATUS_T_L_C (4+0+0)  //-- Thermostat mode - Active Low - Continuous
#define STATUS_CURRENT STATUS_T_L_C



//************** VARIABLES **************
signed char TempH=0;       //-- High Thermo setting
signed char TempL=0;       //-- Low thermo setting
char TempS=0xFF;           //-- Status register setting

char ValueType=THIGH;      //-- Indicates the current displayed type
char EditPosition=0;       //-- Selects current digit to edit
char DisplayBuff[5];       //-- 4 characters on display so 4 in buffer
char count;
union
{
 char Data[4];
 struct
 {
  char Ones;               //-- Editable Areas
  char Tens;
  char Hundreds;
  char Type;               //-- Type of area - High/Low/Clear/Program
 }Area;
}Edit;

char Current;
char temp;

bit  HighLowTemp;           //-- High or Low temperature now displayed
bit  Negative;              //-- Indicates a negative value is present
bit  Flash;                 //-- Flashes the selected Digit if '1'
bit  Programmed;
bit  ClrL;
bit  ClrH;


//************ Functions *************
void ToggleMode(void);        //-- Toggle the Thermostat/1-wire mode
void ReadCurrent(void);       //-- Read data from the chip
void WriteCurrent(void);      //-- Write data to the chip
void DisplayCurrent(void);    //-- Display data on the display
void DoUserIO(void);          //-- Do adjustmets at users request
char GetKeyPressed(void);     //-- Gets key pressed and flashes current digit

void ConvertForDisplay(void); //-- Converts read data for Displaying
void ConvertForUpload(void);  //-- Converts displayed data for writing

//************************************************************************
//                             main
//************************************************************************
void main(void)
{
 DelayMs(50);      //-- Power up delay
	HighLowTemp=1;
	Negative=0;
 Flash=1;
 Programmed=0;
 ClrL=0;
 ClrH=0;
 //-- Init ports --
 SW1_DIR=1;          //-- Keys are inputs
 SW2_DIR=1;
 SPWR=1;             //-- Power to device is on
 SPWR_DIR=0;         //   and set as ouput
 DelayMs(150);       //-- Power ON Delay

 #ifdef _16C64
  OPTION=0xFF;
  INTCON=0x00;
  PSPMODE=0;
 #endif


 ToggleMode();                //-- Force DS1821 into Programming Mode
 DelayMs(200);                //-- Big delay before reading
 
 ReadCurrent();               //-- Read Current Temperature settings
 while(1)
 {
  DisplayCurrent();            //-- Display current settings
  DoUserIO();                  //-- read keys and do what is required
 }
}
//*********** END OF main

//************************************************************************
//                          ToggleMode
//
// Force's DS1821 into Programming Mode.
// Hold DQ pin High, Lower Power Pin, then Toggle DQ 16 times then release
//
//************************************************************************
void ToggleMode(void)
{
 D_PIN=0;
 D_TRIS=1;            //-- Set DQ high (external pull up)

// for(count=0;count<20;count++)
// {
//  DelayMs(250);    //-- .25 sec * 10 = 2.5 Seconds
// }


 SPWR=0;              //-- Lower the power Line
 SPWR_DIR=0;          //   and set as ouput
 DelayMs(_TPC);       //-- Delay for DS1821 to sort out what is happening

 for(count=16;count>0;count--)
 {
  D_TRIS=0;           //-- Lower the  DQ pin
  DelayMs(_TCL);      //-- Minimum delay required for low
  D_TRIS=1;           //-- Raise the pin
  DelayMs(_TCH );     //-- Minimum High Time + Transition time
 }

 DelayMs(_TCP);       //-- Rise time for the Power Pin
 SPWR=1;              //-- Power to device is back on so can communicate
 SPWR_DIR=0;          //   and set as ouput
}
//*********** END OF ToggleMode

//************************************************************************
//                          ReadCurrent
//
// Read Current Temperature settings from the thermo device
//************************************************************************
void ReadCurrent(void)
{
 D_Reset();          //-- Reset the Bus
 D_Write(0xA1);      //-- Issue Read Temp High Command
 TempH=D_Read();     //-- Read the "High Temp"

 D_Reset();          //-- Reset the Bus
 D_Write(0xA2);      //-- Issue Read Temp Low Command
 TempL=D_Read();     //-- Read the "Low Temp"

 D_Reset();          //-- Reset the bus
 D_Write(0xAC);      //-- Issue Read Status Command
 TempS=D_Read();     //-- read the status Byte

}
//*********** END OF ReadCurrent

//************************************************************************
//                          WriteCurrent
//
// Write Current Temperature settings from the thermo device
//************************************************************************
void WriteCurrent(void)
{

 if(TempH >MAXTEMP) TempH=MAXTEMP;
 if(TempL < MINTEMP ) TempL= MINTEMP;

 D_Reset();          //-- Reset the Bus
 D_Write(0x01);      //-- Issue Write Temp High Command
 D_Write(TempH);     //-- Write the "High Temp"

 D_Reset();          //-- Reset the Bus
 D_Write(0x02);      //-- Issue Write Temp Low Command
 D_Write(TempL);     //-- Write the "Low Temp"

 D_Reset();          //-- Reset the bus
 D_Write(0x0C);      //-- Write Write Status Command
 D_Write(STATUS_CURRENT); //-- Write the status Byte
 //D_Write(TempS);     //-- Write the status Byte

}
//*********** END OF ReadCurrent


//************************************************************************
//                        DisplayCurrent
//Display current settings of the Buffer
//************************************************************************
void DisplayCurrent(void)
{
 ConvertForDisplay();
 
 DisplayBuff[0]=0;      //-- Command Byte

 switch(Edit.Area.Hundreds)   //-- Select '0','1' or '-'
 {
  case 0:
  case 1:
   DisplayBuff[2]=Digits[Edit.Area.Hundreds];
   break;
  case T_NEGATIVE:
   DisplayBuff[2]=MINUS;
   break;
  default:
   DisplayBuff[2]=ERROR;
   break;
 }

 DisplayBuff[3]=Digits[Edit.Area.Tens];   //-- The tens digit
 DisplayBuff[4]=Digits[Edit.Area.Ones];   //-- The Ones Digit

 switch(Edit.Area.Type) //-- Current Edit Mode
 {
  case T_HIGH:
   DisplayBuff[1]=THIGH;
   break;
  case T_LOW:
   DisplayBuff[1]=TLOW;
   break;

  case T_CLRL:
  case T_CLRH:
   DisplayBuff[1]=CLRC;
   DisplayBuff[2]=CLRL;
   DisplayBuff[3]=CLRR;
   if(Edit.Area.Type==T_CLRL)
   {
    if(ClrL)
    {
     DisplayBuff[4]=Digits[0];
    }
    else
    {
     DisplayBuff[4]=TLOW;
    }
   }
   else
   {
    if(ClrH)
    {
     DisplayBuff[4]=Digits[0];
    }
    else
    {
     DisplayBuff[4]=THIGH;
    }
   }
   break;


  case T_PROG:
   DisplayBuff[1]=PROGP;
   DisplayBuff[2]=PROGR;
   DisplayBuff[3]=PROGG;
   if(Programmed)
   {
    DisplayBuff[4]=PRGDONE;
   }
   else
   {
    DisplayBuff[4]=MINUS;
   }
   break;
  default:
   DisplayBuff[1]=ERROR;
   break;
 }

 //-- Write the buffer to the LCD module for displaying
 DisplayBuff[0]=0;
 I2C_Send(0x74,DisplayBuff,5);
}
//*********** END OF DisplayCurrent

//************************************************************************
//                          DoUserIO
// read keys and do what is required
//************************************************************************
void DoUserIO(void)
{

 switch(GetKeyPressed())
 {
  //*********** Minus Pressed **************
  case KEY_MINUS:
   switch(Edit.Area.Type)
   {
    case T_CLRL:
     TempL=0;
     ClrL=1;
     break;

    case T_CLRH:
     TempH=0;
     ClrH=1;
     break;

    case T_HIGH:
     TempH--;
     if(TempH < MINTEMP) TempH=MINTEMP;
     Programmed=0;
     ClrH=0;
     break;

    case T_LOW:
     TempL--;
     if(TempL < MINTEMP) TempL=MINTEMP;
     Programmed=0;
     ClrL=0;
     break;

    case T_PROG:
     WriteCurrent();
     Programmed=1;
     break;
   }
   break;

  //*********** Plus Pressed **************
  case KEY_PLUS:
   switch(Edit.Area.Type)
   {
    case T_CLRL:
     TempL=0;
     ClrL=1;
     break;

    case T_CLRH:
     TempH=0;
     ClrH=1;
     break;

    case T_HIGH:
     TempH++;
     if(TempH > MAXTEMP) TempH=MAXTEMP;
     Programmed=0;
     ClrH=0;
     break;

    case T_LOW:
     TempL++;
     if(TempL > MAXTEMP) TempL=MAXTEMP;
     Programmed=0;
     ClrL=0;
     break;

    case T_PROG:
     WriteCurrent();
     Programmed=1;
     break;
   }
   break;

  //*********** Select Pressed **************
  case KEY_SELECT:
   Edit.Area.Type++;
   if(Edit.Area.Type>T_PROG)Edit.Area.Type=0;
   break;
 }
 Current=Edit.Data[EditPosition];        //-- Ensure current data loaded
}
//*********** END OF DoUserIO

//************************************************************************
//GetKeyPressed
//************************************************************************
char GetKeyPressed(void)
{
 char pos,done=0;
 pos=4-EditPosition;
 temp=DisplayBuff[pos];
 while(!done)
 {
  DelayMs(250);
  if(Flash)
  {
   DisplayBuff[pos]=LBLANK;
   I2C_Send(0x74,DisplayBuff,5);  //-- Blank the Digit
  }
  if(!SW1)
  {
   done+=1;
  }
  if(!SW2)
  {
   done+=2;
  }
  DelayMs(250);
  if(Flash)
  {
   DisplayBuff[pos]=temp;
   I2C_Send(0x74,DisplayBuff,5); //-- Reshow the digit
  }
  switch(done)
  {
   case 1:
    if(SW1) done=0;            //-- Check if key still Held Down
    break;
   case 2:
    if(SW2) done=0;            //-- Check if key still Held Down
    break;
   case 3:
    if(SW1 | SW2) done=0;      //-- Check if both keys still Held Down
    break;
   default:
    done=0;
    break;
  }
 }
 return(done);
}
//*********** END OF GetKeyPressed

//************************************************************************
//                       ConvertForDisplay
//************************************************************************
void ConvertForDisplay(void)
{
 signed char temp=0;
 //---- Clear all values to start with -----
 Edit.Area.Hundreds=0;
 Edit.Area.Tens=0;
 Edit.Area.Ones=0;
 //---- Convert values to Editable Digits ----
 switch(Edit.Area.Type)
 {
  case T_HIGH:
   temp=TempH;
   break;

  case T_LOW:
   temp=TempL;
   break;

  default:
   return;
 }
 if(temp <0)
 {
  temp=0-temp;    //-- Convert all values to +ve Digits
  if(temp > 99) temp=99;
  Edit.Area.Hundreds=T_NEGATIVE;
 }
 else
 {
  if(temp >99)
  {
   Edit.Area.Hundreds=(temp/100);
   temp-=(Edit.Area.Hundreds*100);
  }
 }
 if(temp >9)
 {
  Edit.Area.Tens=(temp/10);
  temp-=(Edit.Area.Tens*10);
 }
 Edit.Area.Ones=temp;
 if(Edit.Area.Hundreds > 2)Edit.Area.Hundreds=0;
 if(Edit.Area.Tens > 9)Edit.Area.Tens=9;
 if(Edit.Area.Ones > 9)Edit.Area.Tens=9;
}
//*********** END OF ConvertForDisplay


//************************************************************************
//
//************************************************************************


//*********** END OF

//************************************************************************
//
//************************************************************************










